/**
 * ================================================================
 * BentoScript© Version 1.0 - Bento System for Illustrator [LITE VERSION]
 * Crafted with love by 2Advanced Studios
 * Designer: Eric Jordan • © 2025
 * Websites: 2advanced.com • ericjordan.com
 * Patreon: patreon.com/2advanced
 * ------------------------------------------------
 * LIGHT vs PRO FEATURES
 * Shared (Lite & Pro):
 *  - All preset patterns
 *  - Grid size & gutter
 *  - Auto-fill from Illustrator foreground color
 *  - Black / Gray / White presets
 *  - Rounded corners toggle
 * Pro only:
 *  - Custom Pattern (Simple & JSON)
 *  - Custom RGB (manual R/G/B fields)
 *  - Stroke styling (enable + width + RGB)
 *  - Background frame (enable + padding)
 *  - Quick Gutter Presets
 * ================================================================
 */


(function () {
    'use strict';

    // Ensure an open document and define a shared doc reference
    if (app.documents.length === 0) { alert('Please open a document first.'); return; }
    var doc = app.activeDocument;
// ===========================
    // Embedded Base64 PNG (2advanced Logo)
    // ===========================
    // Supports raw base64 or a data URL (e.g., "data:image/png;base64,....")
    var EMBEDDED_LOGO_BASE64 = "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";

    // ====================================
    // Utility: Decode Base64 to temp file
    // ====================================
    function embeddedLogoFile() {
        try {
            var tmpFolder = Folder.temp;
            var out = File(tmpFolder.fsName + "/bento_logo_embedded.png");
            out.encoding = 'BINARY';
            if (out.exists) out.remove();
            out.open('w');
            (function writeBase64ToFile(b64, f){
                var chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
                var i = 0, c1, c2, c3, e1, e2, e3, e4;
                while (i < b64.length) {
                    e1 = chars.indexOf(b64.charAt(i++));
                    e2 = chars.indexOf(b64.charAt(i++));
                    e3 = chars.indexOf(b64.charAt(i++));
                    e4 = chars.indexOf(b64.charAt(i++));
                    c1 = (e1 << 2) | (e2 >> 4);
                    c2 = ((e2 & 15) << 4) | (e3 >> 2);
                    c3 = ((e3 & 3) << 6) | e4;
                    f.write(String.fromCharCode(c1));
                    if (e3 != 64) f.write(String.fromCharCode(c2));
                    if (e4 != 64) f.write(String.fromCharCode(c3));
                }
            })(EMBEDDED_LOGO_BASE64, out);
            out.close();
            return out;
        } catch (e) { return null; }
    }
    // =========================
    // Color helpers (restored)
    // =========================
    function clamp01(v){ return Math.max(0, Math.min(1, v)); }
    function createColor(rgb){
        var c = new RGBColor();
        c.red = Math.round(rgb[0]*255);
        c.green = Math.round(rgb[1]*255);
        c.blue = Math.round(rgb[2]*255);
        return c;
    }
    function colorToRGB01(c){
        try{
            if (!c) return null;
            if (c.typename === 'RGBColor'){
                return [clamp01(c.red/255), clamp01(c.green/255), clamp01(c.blue/255)];
            } else if (c.typename === 'GrayColor'){
                var v = 1 - (c.gray/100);
                return [clamp01(v), clamp01(v), clamp01(v)];
            } else if (c.typename === 'CMYKColor'){
                var C=c.cyan/100, M=c.magenta/100, Y=c.yellow/100, K=c.black/100;
                var r = 255*(1-C)*(1-K), g = 255*(1-M)*(1-K), b = 255*(1-Y)*(1-K);
                return [clamp01(r/255), clamp01(g/255), clamp01(b/255)];
            } else if (c.typename === 'SpotColor'){
                return colorToRGB01(c.spot.color);
            }
        }catch(e){}
        return null;
    }
    function presetToRGB01(name){
        if (name === 'black') return [0,0,0];
        if (name === 'white') return [1,1,1];
        return [0.12,0.12,0.12]; // gray default
    }
    function safeGetRGB(rgb){
        if (!rgb || rgb.length !== 3) return [0.12,0.12,0.12];
        var r = Number(rgb[0]), g = Number(rgb[1]), b = Number(rgb[2]);
        if (isNaN(r)||isNaN(g)||isNaN(b)) return [0.12,0.12,0.12];
        return [clamp01(r), clamp01(g), clamp01(b)];
    }

    // ---- Lite/Pro gating helpers - LITE VERSION ----
    function allowRGB(){ return false; }           // Custom RGB (Pro only)
    function allowStroke(){ return false; }        // Stroke controls (Pro only)
    function allowBackground(){ return false; }    // Background frame (Pro only)
    function allowCustomPattern(){ return false; } // Custom pattern (Pro only)
    function allowQuickGutter(){ return false; }   // Quick Gutter Presets (Pro only)
    // ---------------------------------

    function getIllustratorForegroundRGB(){

    // Illustrator default stroke color → [r,g,b] in 0..1 (fallback gray)
    function getIllustratorStrokeRGB(){
        try{
            if (app.documents.length === 0) return presetToRGB01('gray');
            var d = app.activeDocument;
            var c = d.defaultStrokeColor;
            var rgb = colorToRGB01(c);
            if (rgb) return rgb;
        }catch(e){}
        return presetToRGB01('gray');
    }
        try{
            if (doc.defaultFillColor){
                var rgb = colorToRGB01(doc.defaultFillColor);
                if (rgb) return rgb;
            }
        }catch(e){}
        try{
            var sel = doc.selection;
            if (sel && sel.length){
                for (var i=0;i<sel.length;i++){
                    if (sel[i].filled){
                        var rgb2 = colorToRGB01(sel[i].fillColor);
                        if (rgb2) return rgb2;
                    }
                }
            }
        }catch(e){}
        return null;
    }
    function presetToRGB01(name){
        if (name === 'black') return [0,0,0];
        if (name === 'white') return [1,1,1];
        return [0.12,0.12,0.12]; // gray default
    }
    function safeGetRGB(rgb){
        if (!rgb || rgb.length !== 3) return [0.12,0.12,0.12];
        var r = Number(rgb[0]), g = Number(rgb[1]), b = Number(rgb[2]);
        if (isNaN(r)||isNaN(g)||isNaN(b)) return [0.12,0.12,0.12];
        return [clamp01(r), clamp01(g), clamp01(b)];
    }

    // =========================
    // Pattern parsing
    // =========================
    function parseSimplePattern(str){
        var rows=str.split('|'), out=[];
        for (var r=0;r<rows.length;r++){
            var cols=rows[r].split(','), prow=[];
            for (var i=0;i<cols.length;i++){ var v=parseInt(cols[i],10); if(!isNaN(v)&&v>0) prow.push({cs:v, rs:1}); }
            out.push(prow);
        }
        return out;
    }
    function parsePattern(str){
        try{
            var p = eval(str); // accepts JSON-like arrays
            if (!(p instanceof Array)) throw 'Pattern must be an array of rows.';
            return p;
        }catch(e){ alert('Custom pattern parse error: '+e); return null; }
    }

    
    // =========================
    // Preset Patterns & Settings (restored)
    // =========================
var bentoPatterns = {
        "Hero Left (3×2)": [
            [ {cs:2, rs:2}, {cs:1, rs:1} ],
            [               {cs:1, rs:1} ]
        ],
        "Hero Right (3×2)": [
            [ {cs:1, rs:1}, {cs:2, rs:2} ],
            [ {cs:1, rs:1} ]
        ],
        "Hero Top (3×2)": [
            [ {cs:3, rs:1} ],
            [ {cs:1, rs:1}, {cs:1, rs:1}, {cs:1, rs:1} ]
        ],
        "Hero Top Wide (4×2)": [
            [ {cs:4, rs:1} ],
            [ {cs:1, rs:1}, {cs:1, rs:1}, {cs:1, rs:1}, {cs:1, rs:1} ]
        ],
        "Hero Bottom (3×2)": [
            [ {cs:1, rs:1}, {cs:1, rs:1}, {cs:1, rs:1} ],
            [ {cs:3, rs:1} ]
        ],
        "Landscape Hero (4×2)": [
            [ {cs:1, rs:1}, {cs:2, rs:1}, {cs:1, rs:1} ],
            [ {cs:1, rs:1}, {cs:1, rs:1}, {cs:1, rs:1}, {cs:1, rs:1} ]
        ],
        "Classic 4×2": [
            [ {cs:1, rs:1}, {cs:1, rs:1}, {cs:1, rs:1}, {cs:1, rs:1} ],
            [ {cs:1, rs:1}, {cs:1, rs:1}, {cs:1, rs:1}, {cs:1, rs:1} ]
        ],
        "Classic 3×3": [
            [ {cs:1, rs:1}, {cs:1, rs:1}, {cs:1, rs:1} ],
            [ {cs:1, rs:1}, {cs:1, rs:1}, {cs:1, rs:1} ],
            [ {cs:1, rs:1}, {cs:1, rs:1}, {cs:1, rs:1} ]
        ]
    };

    var patternNames = [];
    for (var k in bentoPatterns) if (bentoPatterns.hasOwnProperty(k)) patternNames.push(k);

var settings = {
        gridWidth: 1200,
        gridHeight: 600,
        gutter: 24,
        useRoundedCorners: false,
        cornerRadius: 16,
        fillColor: [0.12, 0.12, 0.12],
        strokeColor: [1, 1, 1],
        strokeWidth: 1,
        useAutoFill: true,
        presetColor: 'gray',
        useStroke: false,
        createBackground: false,
        backgroundPadding: 24,
        selectedPattern: patternNames[0],
        customPattern: '',
        simplePattern: ''
    };

// =========================
    // Dialog
    // =========================
    function createDialog() {
        var dialog = new Window('dialog', 'BentoScript\u00A9 - Bento System for Illustrator');

        // --- Embedded Logo + Title + Credits ---
        try {
            var scriptFolder = File($.fileName).parent;
            var logoFile = File(scriptFolder.fsName + "/bento_logo.png");
            var toUse = logoFile.exists ? logoFile : embeddedLogoFile();
            var wrap = dialog.add('group');
            wrap.alignment = 'center';
            wrap.orientation = 'column';
            wrap.margins = [0, 16, 0, 8];
            wrap.alignChildren = 'center';

            if (toUse && toUse.exists) {
                var row = wrap.add('group');
                row.orientation = 'row';
                row.alignChildren = 'left';
                row.spacing = 0; row.margins = 0;
                var img = row.add('image', undefined, toUse);
                img.preferredSize = [128, 128];
                var spacer = row.add('statictext', undefined, ''); spacer.minimumSize.width = 4;
            }

            var titleText = wrap.add('statictext', undefined, 'BentoScript\u00A9 - Bento System for Illustrator');
            try { titleText.graphics.font = ScriptUI.newFont(titleText.graphics.font.name, ScriptUI.FontStyle.BOLD, 15); } catch(_){}var creditsLine1 = wrap.add('statictext', undefined, 'Crafted with love by 2Advanced Studios');
            creditsLine1.alignment = 'center';
            var creditsLine2 = wrap.add('statictext', undefined, 'Designed by Eric Jordan • © 2025');
            creditsLine2.alignment = 'center';
        } catch (e) {}
        dialog.orientation = 'column';
        dialog.alignChildren = 'fill';

        // Size from active artboard
        var ab = doc.artboards[doc.artboards.getActiveArtboardIndex()].artboardRect;
        settings.gridWidth = ab[2] - ab[0];
        settings.gridHeight = ab[1] - ab[3];

        var mainPanel = dialog.add('panel', undefined, 'Grid Settings');
        mainPanel.orientation = 'column';
        mainPanel.alignChildren = 'fill';
        mainPanel.margins = 14;

        var dimRow = mainPanel.add('group');
        dimRow.add('statictext', undefined, 'Grid Size:');
        var widthInput = dimRow.add('edittext', undefined, settings.gridWidth.toString()); widthInput.characters = 8;
        dimRow.add('statictext', undefined, '\u00D7');
        var heightInput = dimRow.add('edittext', undefined, settings.gridHeight.toString()); heightInput.characters = 8;
        dimRow.add('statictext', undefined, 'px');

        var patternRow = mainPanel.add('group');
        patternRow.add('statictext', undefined, 'Preset Pattern:');
        var patternDropdown = patternRow.add('dropdownlist', undefined, patternNames);
        patternDropdown.selection = 0;

        var customPatternRow = mainPanel.add('group');
        var _cpLbl = customPatternRow.add('statictext', undefined, 'Custom Pattern (Pro only):');
        var customPatternBtn = customPatternRow.add('button', undefined, 'Configure Custom Pattern...');
        var customStatusText = customPatternRow.add('statictext', undefined, 'None');
        customStatusText.characters = 20;

        var gutterRow = mainPanel.add('group');
        gutterRow.add('statictext', undefined, 'Gutter:');
        var gutterInput = gutterRow.add('edittext', undefined, settings.gutter.toString()); gutterInput.characters = 4;

        // Quick Gutter Presets (Pro only) - LITE VERSION
        var gutterPresetRow = mainPanel.add('group');
        gutterPresetRow.alignment = 'left';
        var gpLabel = gutterPresetRow.add('statictext', undefined, 'Quick Gutter Presets (Pro only):');
        var gpDrop = gutterPresetRow.add('dropdownlist', undefined, ['Tight (12px)', 'Comfort (24px)', 'Loose (48px)', 'Spacious (64px)', 'Custom']);
        var _syncingGutter = false;
        gpDrop.preferredSize.width = 220;

        // Helper functions
        function getPresetIndex(pixelValue) {
            switch (pixelValue) {
                case 12: return 0; // Tight
                case 24: return 1; // Comfort  
                case 48: return 2; // Loose
                case 64: return 3; // Spacious
                default: return 4; // Custom
            }
        }

        function getPixelsFromIndex(index) {
            switch (index) {
                case 0: return 12; // Tight
                case 1: return 24; // Comfort
                case 2: return 48; // Loose
                case 3: return 64; // Spacious
                default: return null; // Custom - don't change
            }
        }

        function validateGutterInput() {
            var val = parseInt(gutterInput.text, 10);
            if (isNaN(val) || val < 0) val = 24; // Default to comfort
            if (val > 512) val = 512; // Max limit
            return val;
        }

        function updateDropdownFromGutter() {
            if (_syncingGutter) return;
            
            var pixels = validateGutterInput();
            var index = getPresetIndex(pixels);
            
            _syncingGutter = true;
            try {
                gutterInput.text = String(pixels); // Ensure field shows validated value
                gpDrop.selection = index;
            } catch(e) {}
            _syncingGutter = false;
        }

        function updateGutterFromDropdown() {
            if (_syncingGutter) return;
            if (!gpDrop.selection) return;
            
            var index = gpDrop.selection.index;
            var pixels = getPixelsFromIndex(index);
            
            if (pixels !== null) { // Only update for preset values, not custom
                _syncingGutter = true;
                try {
                    gutterInput.text = String(pixels);
                } catch(e) {}
                _syncingGutter = false;
            }
        }

        // Set initial state - dropdown should match current gutter value
        var initialGutter = validateGutterInput();
        gutterInput.text = String(initialGutter); // Ensure field shows correct value
        gpDrop.selection = getPresetIndex(initialGutter);

        // Event handlers - ONLY use onChange, not onChanging
        gpDrop.onChange = function() {
            updateGutterFromDropdown();
        };

        // Only ONE handler for the input field
        gutterInput.onChange = function() {
            updateDropdownFromGutter();
        };

        // Remove any potential onChanging handler
        try {
            gutterInput.onChanging = null;
        } catch(e) {}

        // Apply Pro gating - LITE VERSION
        try {
            var _gpOK = (typeof allowQuickGutter === 'function') ? allowQuickGutter() : true;
            gpDrop.enabled = _gpOK;
            if(!_gpOK) gpDrop.enabled = false;
            if(!_gpOK){
                var gpNote = gutterPresetRow.add('statictext', undefined, '(Pro only)');
                try{
                    var g = gpNote.graphics, f = g.font;
                    gpNote.graphics.font = ScriptUI.newFont(f.name, ScriptUI.FontStyle.ITALIC, 10);
                    var pen = g.newPen(g.PenType.SOLID_COLOR, [0.7,0.7,0.7,1], 1);
                    g.foregroundColor = pen;
                }catch(_){}
            }
        } catch(_) {}

        gutterRow.add('statictext', undefined, 'px');

        // Custom pattern dialog
        function openCustomPatternDialog() {
            var customDialog = new Window('dialog', 'Custom Pattern Configuration');
            customDialog.orientation = 'column';
            customDialog.alignChildren = 'fill';
            customDialog.minimumSize.width = 450;

            var typePanel = customDialog.add('panel', undefined, 'Pattern Type');
            typePanel.orientation = 'column'; typePanel.alignChildren = 'fill'; typePanel.margins = 10;

            var typeGroup = typePanel.add('group');
            var simpleRadio = typeGroup.add('radiobutton', undefined, 'Simple Pattern (e.g., 2,1|1,1,1)');
            var advancedRadio = typeGroup.add('radiobutton', undefined, 'Advanced Pattern (JSON)');

            if (settings.selectedPattern === 'Simple Pattern') simpleRadio.value = true;
            else if (settings.selectedPattern === 'Advanced Pattern (JSON)') advancedRadio.value = true;
            else simpleRadio.value = true;

            var contentArea = customDialog.add('panel', undefined, '');
            contentArea.orientation = 'column';
            contentArea.alignChildren = 'fill';
            contentArea.margins = 10;
            contentArea.preferredSize.height = 200;

            var simpleInput, advancedInput;

            var btnGroup = customDialog.add('group');
            btnGroup.alignment = 'center';
            btnGroup.spacing = 10;
            btnGroup.margins = [0, 10, 0, 10];

            var cancelBtn = btnGroup.add('button', undefined, 'Cancel');
            cancelBtn.preferredSize.width = 80;
            cancelBtn.onClick = function(){ customDialog.close(); };

            var okBtn = btnGroup.add('button', undefined, 'OK');
            okBtn.preferredSize.width = 80;
            okBtn.onClick = function() {
                if (simpleRadio.value) {
                    settings.selectedPattern = 'Simple Pattern';
                    settings.simplePattern = simpleInput ? simpleInput.text : '';
                    customStatusText.text = 'Simple: ' + (settings.simplePattern || 'Empty');
                } else {
                    settings.selectedPattern = 'Advanced Pattern (JSON)';
                    settings.customPattern = advancedInput ? advancedInput.text : '';
                    customStatusText.text = 'JSON: ' + (settings.customPattern ? 'Configured' : 'Empty');
                }
                customDialog.close();
            };

            function updatePanelContent() {
                while (contentArea.children.length > 0) {
                    contentArea.children[0].hide();
                    contentArea.remove(contentArea.children[0]);
                }
                if (simpleRadio.value) {
                    contentArea.text = 'Simple Pattern';
                    simpleInput = contentArea.add('edittext', undefined, settings.simplePattern, { multiline:false });
                    simpleInput.characters = 40;
                    simpleInput.preferredSize.width = 400;
                    contentArea.add('statictext', undefined, "Rows separated by '|' ; col spans separated by commas. Example: 2,1|1,1,1");
                } else {
                    contentArea.text = 'Advanced Pattern (JSON)';
                    contentArea.add('statictext', undefined, 'Cell object format: {cs:colSpan, rs:rowSpan}');
                    var exampleText = contentArea.add('edittext', undefined, [
                        'Examples:',
                        '• Hero Left: [[{"cs":2,"rs":2},{"cs":1,"rs":1}],[{"cs":1,"rs":1}]]',
                        '• Hero Top: [[{"cs":3,"rs":1}],[{"cs":1,"rs":1},{"cs":1,"rs":1},{"cs":1,"rs":1}]]',
                        'Tip: Use null for intentional empty spots. All numbers must be > 0.'
                    ].join('\\n'), { multiline:true, readonly:true });
                    exampleText.preferredSize = [400, 60];
                    advancedInput = contentArea.add('edittext', undefined, settings.customPattern, { multiline:true });
                    advancedInput.preferredSize = [400, 80];
                }
                customDialog.layout.layout(true);
                customDialog.layout.resize();
            }
            simpleRadio.onClick = updatePanelContent;
            advancedRadio.onClick = updatePanelContent;
            updatePanelContent();
            customDialog.show();
        }
        customPatternBtn.onClick = openCustomPatternDialog;

        // Appearance
        var appear = dialog.add('panel', undefined, 'Appearance');
        appear.orientation = 'column';
        appear.alignChildren = 'fill';
        appear.margins = 14;

        var fillSrcGroup = appear.add('group');
        fillSrcGroup.orientation = 'row';
        var autoFillCheck = fillSrcGroup.add('checkbox', undefined, 'Use Illustrator Foreground Fill (auto → gray)');
        autoFillCheck.value = true; // default
        settings.useAutoFill = true; // keep in sync initially

        var presetGroup = appear.add('group'); presetGroup.orientation = 'row';
        presetGroup.add('statictext', undefined, 'Or choose:');
        var rBlack = presetGroup.add('radiobutton', undefined, 'Black');
        var rGray  = presetGroup.add('radiobutton', undefined, 'Gray');
        var rWhite = presetGroup.add('radiobutton', undefined, 'White');
        var rCustom= presetGroup.add('radiobutton', undefined, 'Custom (Pro only)');
        rGray.value = true;

        var fillRow = appear.add('group');
        fillRow.add('statictext', undefined, 'Fill RGB (0–1) (Pro only):');
        var fillR = fillRow.add('edittext', undefined, settings.fillColor[0].toString()); fillR.characters = 4;
        var fillG = fillRow.add('edittext', undefined, settings.fillColor[1].toString()); fillG.characters = 4;
        var fillB = fillRow.add('edittext', undefined, settings.fillColor[2].toString()); fillB.characters = 4;

        // Separator: color controls → rounded corners
        var sepColor = appear.add('panel');
        sepColor.alignment = 'fill';
        sepColor.margins = 0;
        sepColor.minimumSize.height = 1;

        var roundedCheck = appear.add('checkbox', undefined, 'Use Rounded Corners');
        roundedCheck.value = settings.useRoundedCorners;

        var cornerRow = appear.add('group');
        cornerRow.add('statictext', undefined, 'Corner Radius:');
        var cornerInput = cornerRow.add('edittext', undefined, settings.cornerRadius.toString()); cornerInput.characters = 6;
        cornerRow.add('statictext', undefined, 'px');
        cornerInput.enabled = roundedCheck.value;
        roundedCheck.onClick = function(){ cornerInput.enabled = roundedCheck.value; };

        function setRGBBoxesEnabled(on){ try{ fillR.enabled = fillG.enabled = fillB.enabled = !!on; }catch(_){ } }
        function mirrorRGB(rgb){
            if (!fillR || !fillG || !fillB) return;
            if (!rgb || rgb.length !== 3) rgb = [0.12,0.12,0.12];
            var r=Number(rgb[0]), g=Number(rgb[1]), b=Number(rgb[2]);
            if (isNaN(r)||isNaN(g)||isNaN(b)) { r=0.12; g=0.12; b=0.12; }
            fillR.text = r.toFixed(2);
            fillG.text = g.toFixed(2);
            fillB.text = b.toFixed(2);
        }
        function currentPresetName(){ return rBlack.value ? 'black' : (rWhite.value ? 'white' : (rCustom.value ? 'custom' : 'gray')); }
        function presetToMirror(){
            var p = currentPresetName();
            if (p !== 'custom') mirrorRGB(safeGetRGB(presetToRGB01(p)));
        }
        function presetChanged(){
            var customOn = (!autoFillCheck.value && rCustom.value);
            setRGBBoxesEnabled(customOn);
            if (!customOn) presetToMirror();
        }
        function syncPresetEnable(){
            var en = !autoFillCheck.value;
            rBlack.enabled = rGray.enabled = rWhite.enabled = rCustom.enabled = en;
            presetChanged();
        }

        autoFillCheck.onClick = function(){
            settings.useAutoFill = autoFillCheck.value; // <-- keep settings in sync
            syncPresetEnable();
            if (autoFillCheck.value){
                var fg = getIllustratorForegroundRGB();
                if (!fg) fg = presetToRGB01('gray');
                mirrorRGB(safeGetRGB(fg));
            } else {
                presetToMirror();
            }
        
            applyLiteProGates();
        };
        rBlack.onClick = function(){ presetChanged(); applyLiteProGates(); };
        rGray.onClick  = function(){ presetChanged(); applyLiteProGates(); };
        rWhite.onClick = function(){ presetChanged(); applyLiteProGates(); };
        rCustom.onClick= function(){ presetChanged(); applyLiteProGates(); };

        var initRGB = autoFillCheck.value ? (getIllustratorForegroundRGB() || presetToRGB01('gray')) : presetToRGB01(currentPresetName());
        mirrorRGB(safeGetRGB(initRGB));
        syncPresetEnable();

        // Separator: rounded corners → stroke controls
        var sepCorners = appear.add('panel');
        sepCorners.alignment = 'fill';
        sepCorners.margins = 0;
        sepCorners.minimumSize.height = 1;

        var strokeCheck = appear.add('checkbox', undefined, 'Add Stroke (Pro only)');
        strokeCheck.value = settings.useStroke;

        var strokeRow = appear.add('group');
        strokeRow.add('statictext', undefined, 'Stroke Width (Pro only):');
        var strokeInput = strokeRow.add('edittext', undefined, settings.strokeWidth.toString()); strokeInput.characters = 4;
        strokeRow.add('statictext', undefined, 'px');

        var strokeColorRow = appear.add('group');
        strokeColorRow.add('statictext', undefined, 'Stroke RGB (0–1) (Pro only):');
        var strokeR = strokeColorRow.add('edittext', undefined, settings.strokeColor[0].toString()); strokeR.characters = 4;
        var strokeG = strokeColorRow.add('edittext', undefined, settings.strokeColor[1].toString()); strokeG.characters = 4;
        var strokeB = strokeColorRow.add('edittext', undefined, settings.strokeColor[2].toString()); strokeB.characters = 4;

        // Separator: stroke controls → background frame
        var sepStroke = appear.add('panel');
        sepStroke.alignment = 'fill';
        sepStroke.margins = 0;
        sepStroke.minimumSize.height = 1;

        var bgCheck = appear.add('checkbox', undefined, 'Create Background Frame (Pro only)');
        bgCheck.value = settings.createBackground;

        var bgRow = appear.add('group');
        bgRow.add('statictext', undefined, 'Background Padding (Pro only):');
        var bgPaddingInput = bgRow.add('edittext', undefined, settings.backgroundPadding.toString()); bgPaddingInput.characters = 6;

        // Toggle background padding enable state when checkbox changes
        bgCheck.onClick = function(){
            try{
                var bOK = (typeof allowBackground==='function') ? allowBackground() : true;
                bgPaddingInput.enabled = (bOK && bgCheck.value);
                applyLiteProGates && applyLiteProGates();
            }catch(_){}
        };
        bgRow.add('statictext', undefined, 'px');

        // Apply Lite/Pro gates to current UI state
        function applyLiteProGates(){
            try {
                var rgbOK = allowRGB();
                // rCustom is disabled if Lite or Auto-Fill is on
                rCustom.enabled = (rgbOK && !autoFillCheck.value);
                setRGBBoxesEnabled(rgbOK && rCustom.value && !autoFillCheck.value);

                var sOK = allowStroke();
                strokeCheck.enabled = sOK;
                strokeInput.enabled = (sOK && strokeCheck.value);
                strokeR.enabled = strokeG.enabled = strokeB.enabled = (sOK && strokeCheck.value);

                var bOK = allowBackground();
                bgCheck.enabled = bOK;
                bgPaddingInput.enabled = (bOK && bgCheck.value);

                if (!allowCustomPattern()){
                    try { customPatternBtn.enabled = false; customStatusText.text = 'Pro feature'; } catch(_){}
                }
            } catch(_){}
        }

        applyLiteProGates();

        var btns = dialog.add('group'); btns.alignment = 'center';
        btns.add('button', undefined, 'Cancel').onClick = function(){ dialog.close(); };
        btns.add('button', undefined, 'Generate Bento Grid').onClick = function(){
            // Enforce gates in case UI was altered
            var _sOK = allowStroke();
            var _bOK = allowBackground();
            if (!_sOK) { settings.useStroke = false; }
            if (!_bOK) { settings.createBackground = false; }
    
            settings.gridWidth = parseInt(widthInput.text,10);
            settings.gridHeight = parseInt(heightInput.text,10);
            settings.gutter = parseFloat(gutterInput.text);
            settings.useRoundedCorners = roundedCheck.value;
            settings.cornerRadius = settings.useRoundedCorners ? parseFloat(cornerInput.text) : 0;

            var chosenRGB = null;
            if (settings.useAutoFill) {
                chosenRGB = getIllustratorForegroundRGB();
            } else {
                var selectedPreset = currentPresetName();
                if (selectedPreset === 'custom') {
                    var rr = parseFloat(fillR.text), gg = parseFloat(fillG.text), bb = parseFloat(fillB.text);
                    if (isNaN(rr)||isNaN(gg)||isNaN(bb)) chosenRGB = presetToRGB01('gray');
                    else chosenRGB = [clamp01(rr), clamp01(gg), clamp01(bb)];
                } else {
                    chosenRGB = presetToRGB01(selectedPreset);
                }
            }
            if (!chosenRGB) chosenRGB = presetToRGB01('gray');
            settings.fillColor = chosenRGB;

            settings.useStroke = strokeCheck.value;
            settings.strokeWidth = parseFloat(strokeInput.text);
            settings.strokeColor = [parseFloat(strokeR.text), parseFloat(strokeG.text), parseFloat(strokeB.text)];
            settings.createBackground = bgCheck.value;
            settings.backgroundPadding = parseFloat(bgPaddingInput.text);

            // Pattern selection
            if (settings.selectedPattern === 'Simple Pattern' || settings.selectedPattern === 'Advanced Pattern (JSON)') {
                // use previously captured values
            } else {
                settings.selectedPattern = patternDropdown.selection.text;
            }

            if (!(settings.gridWidth > 0 && settings.gridHeight > 0)) { alert('Grid size must be positive.'); return; }
            if (!(settings.gutter >= 0)) { alert('Gutter must be 0 or positive.'); return; }
            if (settings.useRoundedCorners && !(settings.cornerRadius >= 0)) { alert('Corner radius must be 0 or positive.'); return; }

            dialog.close(); generateBentoGrid();
        };

        return dialog;
    }

    // =========================
    // Grid Generation
    // =========================
    function cellKey(r,c){ return r + '_' + c; }

    function generateSpanningGrid(group, startX, startY, pattern, gridW, gridH, gutter){
        var rows=pattern.length, cols=0;
        for (var r=0;r<rows;r++){ var sum=0; for (var j=0;j<pattern[r].length;j++){ sum += (pattern[r][j].cs||1); } if(sum>cols) cols=sum; }
        var cellW=(gridW-gutter*(cols-1))/cols, cellH=(gridH-gutter*(rows-1))/rows;
        var occ={};
        for (var r=0;r<rows;r++){
            var c=0;
            for (var k2=0;k2<pattern[r].length;k2++){
                while(occ[cellKey(r,c)]) c++;
                var cell=pattern[r][k2]; if(!cell){ c++; continue; }
                var cs=cell.cs||1, rs=cell.rs||1;
                var x=startX + c*(cellW+gutter), y=startY - r*(cellH+gutter);
                var w=cellW*cs + gutter*(cs-1), h=cellH*rs + gutter*(rs-1);
                createGridCell(group,x,y,w,h,r,c);
                for (var dr=0;dr<rs;dr++) for (var dc=0;dc<cs;dc++) occ[cellKey(r+dr,c+dc)] = true;
                c += cs;
            }
        }
    }

    function createGridCell(group,x,y,w,h,row,col){
        var it;
        if (settings.useRoundedCorners && settings.cornerRadius>0){
            it = group.pathItems.roundedRectangle(y, x, w, h, settings.cornerRadius, settings.cornerRadius);
        } else {
            it = group.pathItems.rectangle(y, x, w, h);
        }
        it.filled = true; it.fillColor = createColor(settings.fillColor);
        it.stroked = settings.useStroke;
        if (settings.useStroke){ it.strokeColor = createColor(settings.strokeColor); it.strokeWidth = settings.strokeWidth; }
        it.name = 'Cell_'+row+'_'+col;
        return it;
    }

    function createBackground(group,x,y){
        var top=y+settings.backgroundPadding, left=x-settings.backgroundPadding;
        var w=settings.gridWidth+settings.backgroundPadding*2, h=settings.gridHeight+settings.backgroundPadding*2;
        var bg;
        if (settings.useRoundedCorners && settings.cornerRadius>0){
            bg = group.pathItems.roundedRectangle(top, left, w, h, settings.cornerRadius*1.5, settings.cornerRadius*1.5);
        } else { bg = group.pathItems.rectangle(top, left, w, h); }
        bg.filled=true; bg.fillColor=createColor([0.95,0.95,0.95]);
        bg.stroked=settings.useStroke; if (settings.useStroke){ bg.strokeColor=createColor(settings.strokeColor); bg.strokeWidth=settings.strokeWidth; }
        bg.name='Background'; try{ bg.zOrder(ZOrderMethod.SENDTOBACK); }catch(e){}
    }

    function generateBentoGrid(){
        var ab = doc.artboards[doc.artboards.getActiveArtboardIndex()];
        var b = ab.artboardRect, startX=b[0], startY=b[1];

        var g = doc.groupItems.add(); g.name='Bento Grid';
        if (settings.createBackground) createBackground(g,startX,startY);

        var pattern;
        if (settings.selectedPattern === 'Advanced Pattern (JSON)'){
            pattern = parsePattern(settings.customPattern); if (!pattern) return;
        } else if (settings.selectedPattern === 'Simple Pattern'){
            pattern = parseSimplePattern(settings.simplePattern);
        } else { pattern = bentoPatterns[settings.selectedPattern]; }

        generateSpanningGrid(g,startX,startY,pattern,settings.gridWidth,settings.gridHeight,settings.gutter);

        // Center on artboard
        var abCenterX = b[0] + (b[2]-b[0])/2, abCenterY = b[1] + (b[3]-b[1])/2;
        var gb = g.visibleBounds, gCenterX=(gb[0]+gb[2])/2, gCenterY=(gb[1]+gb[3])/2;
        g.left += (abCenterX - gCenterX); g.top += (abCenterY - gCenterY);

        doc.selection = null; g.selected = true;
    }

    // Run
    var dlg = createDialog(); dlg.show();

})();